import { glob } from 'glob';
import { wrappedFs as fs } from './wrapped-fs.js';
import path from 'node:path';
export async function determineFileType(filename) {
    const stat = await fs.lstat(filename);
    if (stat.isFile()) {
        return { type: 'file', stat };
    }
    else if (stat.isDirectory()) {
        return { type: 'directory', stat };
    }
    else if (stat.isSymbolicLink()) {
        return { type: 'link', stat };
    }
    return null;
}
export async function crawl(dir, options) {
    const metadata = {};
    // TODO replace with `fs.glob`
    const crawled = await glob(dir, {
        windowsPathsNoEscape: true,
        ...options,
    });
    const results = await Promise.all(crawled.sort().map(async (filename) => [filename, await determineFileType(filename)]));
    const links = [];
    const filenames = results
        .map(([filename, type]) => {
        if (type) {
            metadata[filename] = type;
            if (type.type === 'link')
                links.push(filename);
        }
        return filename;
    })
        .filter((filename) => {
        // Newer glob can return files inside symlinked directories, to avoid
        // those appearing in archives we need to manually exclude theme here
        const exactLinkIndex = links.findIndex((link) => filename === link);
        return links.every((link, index) => {
            if (index === exactLinkIndex) {
                return true;
            }
            const isFileWithinSymlinkDir = filename.startsWith(link);
            // symlink may point outside the directory: https://github.com/electron/asar/issues/303
            const relativePath = path.relative(link, path.dirname(filename));
            return !isFileWithinSymlinkDir || relativePath.startsWith('..');
        });
    });
    return [filenames, metadata];
}
//# sourceMappingURL=crawlfs.js.map