﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/AttributeBooleanValue.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class ModifyVolumeAttributeRequest : public EC2Request {
 public:
  AWS_EC2_API ModifyVolumeAttributeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyVolumeAttribute"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Indicates whether the volume should be auto-enabled for I/O operations.</p>
   */
  inline const AttributeBooleanValue& GetAutoEnableIO() const { return m_autoEnableIO; }
  inline bool AutoEnableIOHasBeenSet() const { return m_autoEnableIOHasBeenSet; }
  template <typename AutoEnableIOT = AttributeBooleanValue>
  void SetAutoEnableIO(AutoEnableIOT&& value) {
    m_autoEnableIOHasBeenSet = true;
    m_autoEnableIO = std::forward<AutoEnableIOT>(value);
  }
  template <typename AutoEnableIOT = AttributeBooleanValue>
  ModifyVolumeAttributeRequest& WithAutoEnableIO(AutoEnableIOT&& value) {
    SetAutoEnableIO(std::forward<AutoEnableIOT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the volume.</p>
   */
  inline const Aws::String& GetVolumeId() const { return m_volumeId; }
  inline bool VolumeIdHasBeenSet() const { return m_volumeIdHasBeenSet; }
  template <typename VolumeIdT = Aws::String>
  void SetVolumeId(VolumeIdT&& value) {
    m_volumeIdHasBeenSet = true;
    m_volumeId = std::forward<VolumeIdT>(value);
  }
  template <typename VolumeIdT = Aws::String>
  ModifyVolumeAttributeRequest& WithVolumeId(VolumeIdT&& value) {
    SetVolumeId(std::forward<VolumeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline ModifyVolumeAttributeRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  AttributeBooleanValue m_autoEnableIO;

  Aws::String m_volumeId;

  bool m_dryRun{false};
  bool m_autoEnableIOHasBeenSet = false;
  bool m_volumeIdHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
