﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/xray/XRay_EXPORTS.h>
#include <aws/xray/model/IndexingRuleValue.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace XRay {
namespace Model {

/**
 * <p> Rule used to determine the server-side sampling rate for spans ingested
 * through the CloudWatchLogs destination and indexed by X-Ray. </p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/xray-2016-04-12/IndexingRule">AWS
 * API Reference</a></p>
 */
class IndexingRule {
 public:
  AWS_XRAY_API IndexingRule() = default;
  AWS_XRAY_API IndexingRule(Aws::Utils::Json::JsonView jsonValue);
  AWS_XRAY_API IndexingRule& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_XRAY_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p> The name of the indexing rule. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  IndexingRule& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Displays when the rule was last modified, in Unix time seconds. </p>
   */
  inline const Aws::Utils::DateTime& GetModifiedAt() const { return m_modifiedAt; }
  inline bool ModifiedAtHasBeenSet() const { return m_modifiedAtHasBeenSet; }
  template <typename ModifiedAtT = Aws::Utils::DateTime>
  void SetModifiedAt(ModifiedAtT&& value) {
    m_modifiedAtHasBeenSet = true;
    m_modifiedAt = std::forward<ModifiedAtT>(value);
  }
  template <typename ModifiedAtT = Aws::Utils::DateTime>
  IndexingRule& WithModifiedAt(ModifiedAtT&& value) {
    SetModifiedAt(std::forward<ModifiedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The indexing rule. </p>
   */
  inline const IndexingRuleValue& GetRule() const { return m_rule; }
  inline bool RuleHasBeenSet() const { return m_ruleHasBeenSet; }
  template <typename RuleT = IndexingRuleValue>
  void SetRule(RuleT&& value) {
    m_ruleHasBeenSet = true;
    m_rule = std::forward<RuleT>(value);
  }
  template <typename RuleT = IndexingRuleValue>
  IndexingRule& WithRule(RuleT&& value) {
    SetRule(std::forward<RuleT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::Utils::DateTime m_modifiedAt{};

  IndexingRuleValue m_rule;
  bool m_nameHasBeenSet = false;
  bool m_modifiedAtHasBeenSet = false;
  bool m_ruleHasBeenSet = false;
};

}  // namespace Model
}  // namespace XRay
}  // namespace Aws
