﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/mediaconnect/MediaConnectRequest.h>
#include <aws/mediaconnect/MediaConnect_EXPORTS.h>
#include <aws/mediaconnect/model/SetSourceRequest.h>

#include <utility>

namespace Aws {
namespace MediaConnect {
namespace Model {

/**
 */
class AddFlowSourcesRequest : public MediaConnectRequest {
 public:
  AWS_MEDIACONNECT_API AddFlowSourcesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddFlowSources"; }

  AWS_MEDIACONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) of the flow that you want to update.</p>
   */
  inline const Aws::String& GetFlowArn() const { return m_flowArn; }
  inline bool FlowArnHasBeenSet() const { return m_flowArnHasBeenSet; }
  template <typename FlowArnT = Aws::String>
  void SetFlowArn(FlowArnT&& value) {
    m_flowArnHasBeenSet = true;
    m_flowArn = std::forward<FlowArnT>(value);
  }
  template <typename FlowArnT = Aws::String>
  AddFlowSourcesRequest& WithFlowArn(FlowArnT&& value) {
    SetFlowArn(std::forward<FlowArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A list of sources that you want to add to the flow.</p>
   */
  inline const Aws::Vector<SetSourceRequest>& GetSources() const { return m_sources; }
  inline bool SourcesHasBeenSet() const { return m_sourcesHasBeenSet; }
  template <typename SourcesT = Aws::Vector<SetSourceRequest>>
  void SetSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources = std::forward<SourcesT>(value);
  }
  template <typename SourcesT = Aws::Vector<SetSourceRequest>>
  AddFlowSourcesRequest& WithSources(SourcesT&& value) {
    SetSources(std::forward<SourcesT>(value));
    return *this;
  }
  template <typename SourcesT = SetSourceRequest>
  AddFlowSourcesRequest& AddSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources.emplace_back(std::forward<SourcesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_flowArn;

  Aws::Vector<SetSourceRequest> m_sources;
  bool m_flowArnHasBeenSet = false;
  bool m_sourcesHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaConnect
}  // namespace Aws
