﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>
#include <aws/events/model/Target.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class PutTargetsRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API PutTargetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutTargets"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the rule.</p>
   */
  inline const Aws::String& GetRule() const { return m_rule; }
  inline bool RuleHasBeenSet() const { return m_ruleHasBeenSet; }
  template <typename RuleT = Aws::String>
  void SetRule(RuleT&& value) {
    m_ruleHasBeenSet = true;
    m_rule = std::forward<RuleT>(value);
  }
  template <typename RuleT = Aws::String>
  PutTargetsRequest& WithRule(RuleT&& value) {
    SetRule(std::forward<RuleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the event bus associated with the rule. If you omit this,
   * the default event bus is used.</p>
   */
  inline const Aws::String& GetEventBusName() const { return m_eventBusName; }
  inline bool EventBusNameHasBeenSet() const { return m_eventBusNameHasBeenSet; }
  template <typename EventBusNameT = Aws::String>
  void SetEventBusName(EventBusNameT&& value) {
    m_eventBusNameHasBeenSet = true;
    m_eventBusName = std::forward<EventBusNameT>(value);
  }
  template <typename EventBusNameT = Aws::String>
  PutTargetsRequest& WithEventBusName(EventBusNameT&& value) {
    SetEventBusName(std::forward<EventBusNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The targets to update or add to the rule.</p>
   */
  inline const Aws::Vector<Target>& GetTargets() const { return m_targets; }
  inline bool TargetsHasBeenSet() const { return m_targetsHasBeenSet; }
  template <typename TargetsT = Aws::Vector<Target>>
  void SetTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets = std::forward<TargetsT>(value);
  }
  template <typename TargetsT = Aws::Vector<Target>>
  PutTargetsRequest& WithTargets(TargetsT&& value) {
    SetTargets(std::forward<TargetsT>(value));
    return *this;
  }
  template <typename TargetsT = Target>
  PutTargetsRequest& AddTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets.emplace_back(std::forward<TargetsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_rule;

  Aws::String m_eventBusName;

  Aws::Vector<Target> m_targets;
  bool m_ruleHasBeenSet = false;
  bool m_eventBusNameHasBeenSet = false;
  bool m_targetsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
