﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeartifact/CodeArtifactRequest.h>
#include <aws/codeartifact/CodeArtifact_EXPORTS.h>
#include <aws/codeartifact/model/Tag.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CodeArtifact {
namespace Model {

/**
 */
class CreatePackageGroupRequest : public CodeArtifactRequest {
 public:
  AWS_CODEARTIFACT_API CreatePackageGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePackageGroup"; }

  AWS_CODEARTIFACT_API Aws::String SerializePayload() const override;

  AWS_CODEARTIFACT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> The name of the domain in which you want to create a package group. </p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  CreatePackageGroupRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The 12-digit account number of the Amazon Web Services account that owns the
   * domain. It does not include dashes or spaces. </p>
   */
  inline const Aws::String& GetDomainOwner() const { return m_domainOwner; }
  inline bool DomainOwnerHasBeenSet() const { return m_domainOwnerHasBeenSet; }
  template <typename DomainOwnerT = Aws::String>
  void SetDomainOwner(DomainOwnerT&& value) {
    m_domainOwnerHasBeenSet = true;
    m_domainOwner = std::forward<DomainOwnerT>(value);
  }
  template <typename DomainOwnerT = Aws::String>
  CreatePackageGroupRequest& WithDomainOwner(DomainOwnerT&& value) {
    SetDomainOwner(std::forward<DomainOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pattern of the package group to create. The pattern is also the
   * identifier of the package group. </p>
   */
  inline const Aws::String& GetPackageGroup() const { return m_packageGroup; }
  inline bool PackageGroupHasBeenSet() const { return m_packageGroupHasBeenSet; }
  template <typename PackageGroupT = Aws::String>
  void SetPackageGroup(PackageGroupT&& value) {
    m_packageGroupHasBeenSet = true;
    m_packageGroup = std::forward<PackageGroupT>(value);
  }
  template <typename PackageGroupT = Aws::String>
  CreatePackageGroupRequest& WithPackageGroup(PackageGroupT&& value) {
    SetPackageGroup(std::forward<PackageGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The contact information for the created package group. </p>
   */
  inline const Aws::String& GetContactInfo() const { return m_contactInfo; }
  inline bool ContactInfoHasBeenSet() const { return m_contactInfoHasBeenSet; }
  template <typename ContactInfoT = Aws::String>
  void SetContactInfo(ContactInfoT&& value) {
    m_contactInfoHasBeenSet = true;
    m_contactInfo = std::forward<ContactInfoT>(value);
  }
  template <typename ContactInfoT = Aws::String>
  CreatePackageGroupRequest& WithContactInfo(ContactInfoT&& value) {
    SetContactInfo(std::forward<ContactInfoT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A description of the package group. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreatePackageGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more tag key-value pairs for the package group.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreatePackageGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreatePackageGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;

  Aws::String m_domainOwner;

  Aws::String m_packageGroup;

  Aws::String m_contactInfo;

  Aws::String m_description;

  Aws::Vector<Tag> m_tags;
  bool m_domainHasBeenSet = false;
  bool m_domainOwnerHasBeenSet = false;
  bool m_packageGroupHasBeenSet = false;
  bool m_contactInfoHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeArtifact
}  // namespace Aws
